## 🐳 Ejemplos en `Dockerfile`

### ✅ Ejemplo 1: API en Node.js

```Dockerfile
FROM node:20

WORKDIR /app
COPY . .

RUN npm install
EXPOSE 3000

HEALTHCHECK --interval=30s --timeout=5s --start-period=10s --retries=3 \
  CMD curl -f http://localhost:3000/health || exit 1

CMD ["npm", "start"]
```

> Aquí se asume que el endpoint `/health` devuelve 200 OK si la app está funcionando.

---

### ✅ Ejemplo 2: PostgreSQL

```Dockerfile
FROM postgres:16

HEALTHCHECK --interval=10s --timeout=3s --retries=5 \
  CMD pg_isready -U postgres || exit 1
```

> `pg_isready` es una utilidad que ya viene en la imagen oficial de PostgreSQL y comprueba si se puede conectar a la base de datos.

---

## 🧩 Ejemplos en `docker-compose.yml`

### ✅ Ejemplo 3: API en Rust

```yaml
version: "3.8"
services:
  rust-api:
    image: mi-api-rust
    ports:
      - "8080:8080"
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8080/health"]
      interval: 30s
      timeout: 5s
      retries: 3
      start_period: 10s
```

> Este sería para una API escrita en Rust usando Axum, por ejemplo.

---

### ✅ Ejemplo 4: Frontend en React

```yaml
services:
  frontend:
    image: mi-frontend-react
    ports:
      - "3000:80"
    healthcheck:
      test: ["CMD-SHELL", "wget -q --spider http://localhost || exit 1"]
      interval: 20s
      timeout: 3s
      retries: 2
```

> Utiliza `wget` para comprobar si el frontend devuelve una respuesta correcta.

---

### ✅ Ejemplo 5: Redis

```yaml
services:
  redis:
    image: redis:7
    ports:
      - "6379:6379"
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 10s
      timeout: 3s
      retries: 3
```

> Redis responde con "PONG" si está funcionando correctamente.

---

### 🧠 Extra: Comprobación del estado desde terminal

Después de definir un healthcheck, puedes consultar el estado con:

```bash
docker ps
```

Verás una columna llamada `STATUS` donde aparecerá `healthy`, `unhealthy` o `starting`.

También puedes ver más información detallada con:

```bash
docker inspect <nombre_contenedor>
```

Y filtrar directamente el estado con:

```bash
docker inspect --format='{{json .State.Health}}' <nombre_contenedor>
```